
import React, { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { Button } from '@/components/ui/button';
import { ArrowLeft, RefreshCw, Users } from 'lucide-react';
import PelangganForm from '@/components/PelangganForm';
import PelangganTable from '@/components/PelangganTable';
import SearchPelanggan from '@/components/SearchPelanggan';
import { getPelanggan } from '@/lib/supabase';

const PelangganPage = ({ onBack }) => {
  const [loading, setLoading] = useState(true);
  const [data, setData] = useState([]);
  const [filteredData, setFilteredData] = useState([]);
  const [editingPelanggan, setEditingPelanggan] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  useEffect(() => {
    fetchPelanggan();
  }, [refreshTrigger]);

  const fetchPelanggan = async () => {
    setLoading(true);
    try {
      const { data: pelanggan, error } = await getPelanggan();
      if (error) throw error;
      setData(pelanggan || []);
      setFilteredData(pelanggan || []);
    } catch (error) {
      console.error('Error fetching pelanggan:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleEdit = (pelanggan) => {
    setEditingPelanggan(pelanggan);
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const handleCancelEdit = () => {
    setEditingPelanggan(null);
  };

  const handleSuccess = () => {
    setEditingPelanggan(null);
    setRefreshTrigger(prev => prev + 1);
  };

  const handleRefresh = () => {
    setRefreshTrigger(prev => prev + 1);
  };

  const handleFilter = (filtered) => {
    setFilteredData(filtered);
  };

  return (
    <div className="space-y-6">
      <motion.div
        initial={{ opacity: 0, y: -10 }}
        animate={{ opacity: 1, y: 0 }}
        className="flex items-center justify-between"
      >
        <div className="flex items-center gap-4">
          {onBack && (
            <Button variant="ghost" size="icon" onClick={onBack} className="text-blue-100 hover:text-white hover:bg-blue-800">
              <ArrowLeft className="h-6 w-6" />
            </Button>
          )}
          <div>
            <h1 className="text-2xl font-bold text-white flex items-center gap-2">
              <Users className="h-6 w-6" />
              Kelola Pelanggan
            </h1>
            <p className="text-blue-200 text-sm">Manajemen data pelanggan Bakulan</p>
          </div>
        </div>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3 }}
      >
        <PelangganForm
          editingPelanggan={editingPelanggan}
          onSuccess={handleSuccess}
          onCancel={editingPelanggan ? handleCancelEdit : undefined}
        />
      </motion.div>

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3, delay: 0.1 }}
        className="flex flex-col sm:flex-row gap-4 items-center justify-between bg-white/10 p-4 rounded-lg backdrop-blur-sm"
      >
        <SearchPelanggan data={data} onFilter={handleFilter} />
        
        <div className="flex gap-2 w-full sm:w-auto">
          <Button
            onClick={handleRefresh}
            variant="outline"
            className="w-full sm:w-auto border-blue-300 text-blue-600 hover:bg-blue-50"
          >
            <RefreshCw className="h-4 w-4 mr-2" />
            Refresh
          </Button>
        </div>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.3, delay: 0.2 }}
      >
        <PelangganTable
          data={filteredData}
          loading={loading}
          onEdit={handleEdit}
          onRefresh={handleSuccess}
        />
      </motion.div>
    </div>
  );
};

export default PelangganPage;
