
import React, { useEffect, useState } from 'react';
import { motion } from 'framer-motion';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Package, ShoppingCart, Users, FileText, TrendingUp, Archive } from 'lucide-react';
import { getInventori, getPelanggan, getTransaksiToday } from '@/lib/supabase';
import { useToast } from '@/components/ui/use-toast';

const MenuPage = ({ onNavigate }) => {
  const { toast } = useToast();
  const [stats, setStats] = useState({
    barang: 0,
    pelanggan: 0,
    penjualanToday: 0
  });

  useEffect(() => {
    const fetchStats = async () => {
      try {
        const [inv, pel, trans] = await Promise.all([
          getInventori(),
          getPelanggan(),
          getTransaksiToday()
        ]);

        const totalPenjualan = trans.data?.reduce((acc, curr) => acc + parseFloat(curr.total_harga || 0), 0) || 0;

        setStats({
          barang: inv.data?.length || 0,
          pelanggan: pel.data?.length || 0,
          penjualanToday: totalPenjualan
        });
      } catch (error) {
        console.error("Failed to fetch dashboard stats", error);
      }
    };

    fetchStats();
  }, []);

  const formatRupiah = (amount) => {
    return new Intl.NumberFormat('id-ID', {
      style: 'currency',
      currency: 'IDR',
      minimumFractionDigits: 0,
    }).format(amount);
  };

  const handleLaporanClick = () => {
    toast({
      title: "Segera Hadir",
      description: "Fitur Laporan sedang dalam pengembangan! 🚀",
    });
  };

  const menuItems = [
    {
      id: 'inventori',
      title: 'Inventori',
      icon: <Package className="h-8 w-8 mb-2" />,
      desc: 'Kelola stok barang',
      color: 'bg-blue-600 hover:bg-blue-700',
      action: () => onNavigate('inventori')
    },
    {
      id: 'transaksi',
      title: 'Transaksi',
      icon: <ShoppingCart className="h-8 w-8 mb-2" />,
      desc: 'Catat penjualan',
      color: 'bg-green-600 hover:bg-green-700',
      action: () => onNavigate('transaksi')
    },
    {
      id: 'pelanggan',
      title: 'Pelanggan',
      icon: <Users className="h-8 w-8 mb-2" />,
      desc: 'Data pelanggan',
      color: 'bg-purple-600 hover:bg-purple-700',
      action: () => onNavigate('pelanggan')
    },
    {
      id: 'laporan',
      title: 'Laporan',
      icon: <FileText className="h-8 w-8 mb-2" />,
      desc: 'Lihat rekapitulasi',
      color: 'bg-orange-600 hover:bg-orange-700',
      action: handleLaporanClick
    }
  ];

  return (
    <div className="space-y-8">
      {/* Header Section */}
      <motion.div
        initial={{ opacity: 0, y: -20 }}
        animate={{ opacity: 1, y: 0 }}
        className="text-center text-white space-y-2 mb-8"
      >
        <h1 className="text-4xl md:text-5xl font-bold tracking-tight">
          BAKULAN
        </h1>
        <p className="text-xl text-blue-100 font-light">
          Aplikasi Nyatet Dagangan
        </p>
      </motion.div>

      {/* Stats Section */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <motion.div
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ delay: 0.1 }}
        >
          <Card className="bg-white/10 backdrop-blur-md border-white/20 text-white">
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-blue-100">
                Total Barang
              </CardTitle>
              <Archive className="h-4 w-4 text-blue-200" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{stats.barang}</div>
              <p className="text-xs text-blue-200">Item terdaftar</p>
            </CardContent>
          </Card>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ delay: 0.2 }}
        >
          <Card className="bg-white/10 backdrop-blur-md border-white/20 text-white">
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-green-100">
                Penjualan Hari Ini
              </CardTitle>
              <TrendingUp className="h-4 w-4 text-green-200" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{formatRupiah(stats.penjualanToday)}</div>
              <p className="text-xs text-green-200">Total omset</p>
            </CardContent>
          </Card>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ delay: 0.3 }}
        >
          <Card className="bg-white/10 backdrop-blur-md border-white/20 text-white">
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium text-purple-100">
                Total Pelanggan
              </CardTitle>
              <Users className="h-4 w-4 text-purple-200" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{stats.pelanggan}</div>
              <p className="text-xs text-purple-200">Orang terdaftar</p>
            </CardContent>
          </Card>
        </motion.div>
      </div>

      {/* Menu Grid */}
      <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 mt-8">
        {menuItems.map((item, index) => (
          <motion.div
            key={item.id}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4 + (index * 0.1) }}
          >
            <Button
              onClick={item.action}
              className={`w-full h-32 flex flex-col items-center justify-center rounded-xl shadow-lg transition-all transform hover:scale-105 ${item.color}`}
            >
              {item.icon}
              <span className="text-xl font-bold">{item.title}</span>
              <span className="text-sm opacity-90 font-normal mt-1">{item.desc}</span>
            </Button>
          </motion.div>
        ))}
      </div>
    </div>
  );
};

export default MenuPage;
