
import { createClient } from '@supabase/supabase-js';
import bcrypt from 'bcryptjs';

// Environment Variable Validation
const supabaseUrl = import.meta.env.VITE_SUPABASE_URL;
const supabaseAnonKey = import.meta.env.VITE_SUPABASE_ANON_KEY;

// Check configuration immediately
const isConfigured = Boolean(
  supabaseUrl && 
  supabaseAnonKey && 
  supabaseUrl.startsWith('http') && 
  supabaseAnonKey.length > 20
);

if (isConfigured) {
  console.log('✅ Supabase configuration detected.');
} else {
  console.error('❌ Supabase configuration missing or invalid. Check your .env file.');
  console.log('Required: VITE_SUPABASE_URL (must start with http/https) and VITE_SUPABASE_ANON_KEY');
}

// Export configuration status for UI
export const isSupabaseConfigured = () => isConfigured;

// Initialize Supabase Client
let supabaseClient = null;

try {
  if (isConfigured) {
    supabaseClient = createClient(supabaseUrl, supabaseAnonKey);
  } else {
    // Create a dummy client that returns errors for all operations to prevent crashes
    // while keeping the app interactive enough to show the warning banner
    const mockAsync = async () => ({ data: null, error: { message: "Supabase belum dikonfigurasi. Cek .env file." } });
    const mockChain = () => ({ 
      select: () => ({ 
        eq: () => ({ single: mockAsync, order: mockAsync }),
        order: mockAsync,
        gte: () => ({ order: mockAsync }),
        insert: mockAsync,
        update: mockAsync,
        delete: mockAsync,
        single: mockAsync
      }),
      insert: () => ({ select: () => ({ single: mockAsync }) }),
      update: () => ({ eq: () => ({ select: () => ({ single: mockAsync }) }) }),
      delete: () => ({ eq: mockAsync })
    });

    supabaseClient = {
      from: mockChain,
      auth: {
        signUp: mockAsync,
        signInWithPassword: mockAsync,
        signOut: mockAsync
      }
    };
  }
} catch (error) {
  console.error("Failed to initialize Supabase client:", error);
}

export const supabase = supabaseClient;

// ==================== AUTH FUNCTIONS ====================

export const hashPassword = async (password) => {
  try {
    const salt = await bcrypt.genSalt(10);
    return await bcrypt.hash(password, salt);
  } catch (error) {
    console.error("Error hashing password:", error);
    throw new Error("Gagal memproses password");
  }
};

export const verifyPassword = async (password, hash) => {
  try {
    return await bcrypt.compare(password, hash);
  } catch (error) {
    console.error("Error verifying password:", error);
    return false;
  }
};

export const registerUser = async (userData) => {
  if (!isConfigured) return { data: null, error: { message: "Konfigurasi Supabase tidak ditemukan" } };
  
  try {
    const hashedPassword = await hashPassword(userData.password);
    
    // 1. Create User
    const { data: user, error: userError } = await supabase
      .from('users')
      .insert([{
        username: userData.username,
        email: userData.email,
        nomor_telepon: userData.nomor_telepon,
        password_hash: hashedPassword,
        role: 'kasir', // Default role
        status: 'aktif'
      }])
      .select()
      .single();

    if (userError) throw userError;

    // 2. Create Default Access
    const accessRights = [
      { user_id: user.id, modul: 'transaksi', akses: true },
      { user_id: user.id, modul: 'inventori', akses: false },
      { user_id: user.id, modul: 'pelanggan', akses: false },
      { user_id: user.id, modul: 'laporan', akses: false }
    ];

    const { error: accessError } = await supabase
      .from('user_akses')
      .insert(accessRights);

    if (accessError) console.error("Error creating default access:", accessError);

    return { data: user, error: null };
  } catch (error) {
    return { data: null, error };
  }
};

export const loginUser = async (identifier, password) => {
  if (!isConfigured) return { data: null, error: { message: "Konfigurasi Supabase tidak ditemukan" } };

  try {
    const isEmail = identifier.includes('@');
    const column = isEmail ? 'email' : 'nomor_telepon';

    const { data: user, error } = await supabase
      .from('users')
      .select('*')
      .eq(column, identifier)
      .single();

    if (error || !user) {
      return { data: null, error: { message: 'User tidak ditemukan' } };
    }

    if (user.status !== 'aktif') {
      return { data: null, error: { message: 'Akun nonaktif' } };
    }

    const validPassword = await verifyPassword(password, user.password_hash);
    if (!validPassword) {
      return { data: null, error: { message: 'Password salah' } };
    }

    const { data: access } = await supabase
      .from('user_akses')
      .select('*')
      .eq('user_id', user.id);

    return { data: { ...user, access }, error: null };
  } catch (error) {
    return { data: null, error };
  }
};

// ==================== INVENTORI FUNCTIONS ====================

export const createInventori = async (data) => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data: result, error } = await supabase
      .from('inventori')
      .insert([{
        nama_barang: data.nama_barang,
        harga: parseFloat(data.harga),
        stok: parseInt(data.stok)
      }])
      .select()
      .single();
    return { data: result, error };
  } catch (error) {
    return { data: null, error };
  }
};

export const getInventori = async () => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data, error } = await supabase
      .from('inventori')
      .select('*')
      .order('created_at', { ascending: false });
    return { data, error };
  } catch (error) {
    return { data: null, error };
  }
};

export const updateInventori = async (id, data) => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const updateData = {};
    if (data.nama_barang !== undefined) updateData.nama_barang = data.nama_barang;
    if (data.harga !== undefined) updateData.harga = parseFloat(data.harga);
    if (data.stok !== undefined) updateData.stok = parseInt(data.stok);

    const { data: result, error } = await supabase
      .from('inventori')
      .update(updateData)
      .eq('id', id)
      .select()
      .single();
    return { data: result, error };
  } catch (error) {
    return { data: null, error };
  }
};

export const deleteInventori = async (id) => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data, error } = await supabase
      .from('inventori')
      .delete()
      .eq('id', id);
    return { data, error };
  } catch (error) {
    return { data: null, error };
  }
};

// ==================== TRANSAKSI FUNCTIONS ====================

export const createTransaksi = async (data) => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    // 1. Check stock
    const { data: barang, error: barangError } = await supabase
      .from('inventori')
      .select('stok')
      .eq('id', data.barang_id)
      .single();

    if (barangError) return { data: null, error: barangError };
    if (!barang || barang.stok < data.jumlah) {
      return { data: null, error: { message: `Stok tidak cukup. Sisa: ${barang?.stok || 0}` } };
    }

    // 2. Create Transaction
    const { data: transaksiResult, error: transaksiError } = await supabase
      .from('transaksi')
      .insert([{
        barang_id: data.barang_id,
        jumlah: parseInt(data.jumlah),
        harga_satuan: parseFloat(data.harga_satuan),
        total_harga: parseFloat(data.total_harga),
        tanggal_transaksi: data.tanggal_transaksi
      }])
      .select()
      .single();

    if (transaksiError) return { data: null, error: transaksiError };

    // 3. Deduct Stock
    const newStok = barang.stok - data.jumlah;
    const { error: updateError } = await supabase
      .from('inventori')
      .update({ stok: newStok })
      .eq('id', data.barang_id);

    if (updateError) {
      await supabase.from('transaksi').delete().eq('id', transaksiResult.id);
      return { data: null, error: updateError };
    }

    return { data: transaksiResult, error: null };
  } catch (error) {
    return { data: null, error };
  }
};

export const getTransaksi = async () => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data, error } = await supabase
      .from('transaksi')
      .select(`
        *,
        inventori:barang_id (nama_barang)
      `)
      .order('tanggal_transaksi', { ascending: false });
    return { data, error };
  } catch (error) {
    return { data: null, error };
  }
};

export const getTransaksiToday = async () => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    const todayISO = today.toISOString();

    const { data, error } = await supabase
      .from('transaksi')
      .select(`
        *,
        inventori:barang_id (nama_barang)
      `)
      .gte('tanggal_transaksi', todayISO)
      .order('tanggal_transaksi', { ascending: false });
    return { data, error };
  } catch (error) {
    return { data: null, error };
  }
};

// ==================== PELANGGAN FUNCTIONS ====================

export const createPelanggan = async (data) => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data: result, error } = await supabase
      .from('pelanggan')
      .insert([{
        nama_pelanggan: data.nama_pelanggan,
        nomor_telepon: data.nomor_telepon,
        alamat: data.alamat,
        email: data.email || null
      }])
      .select()
      .single();
    return { data: result, error };
  } catch (error) {
    return { data: null, error };
  }
};

export const getPelanggan = async () => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data, error } = await supabase
      .from('pelanggan')
      .select('*')
      .order('created_at', { ascending: false });
    return { data, error };
  } catch (error) {
    return { data: null, error };
  }
};

export const updatePelanggan = async (id, data) => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data: result, error } = await supabase
      .from('pelanggan')
      .update(data)
      .eq('id', id)
      .select()
      .single();
    return { data: result, error };
  } catch (error) {
    return { data: null, error };
  }
};

export const deletePelanggan = async (id) => {
  if (!isConfigured) return { data: null, error: { message: "Supabase Offline" } };
  try {
    const { data, error } = await supabase
      .from('pelanggan')
      .delete()
      .eq('id', id);
    return { data, error };
  } catch (error) {
    return { data: null, error };
  }
};
