
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { useToast } from '@/components/ui/use-toast';
import { getInventori, createTransaksi, isSupabaseConfigured } from '@/lib/supabase';
import { Loader2, ShoppingCart } from 'lucide-react';

const TransaksiForm = ({ onSuccess }) => {
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);
  const [inventoriList, setInventoriList] = useState([]);
  const [selectedBarang, setSelectedBarang] = useState(null);
  const [formData, setFormData] = useState({
    barang_id: '',
    jumlah: '',
    harga_satuan: '',
    total_harga: '',
    tanggal_transaksi: new Date().toISOString().slice(0, 16)
  });

  useEffect(() => {
    fetchInventori();
  }, []);

  const fetchInventori = async () => {
    if (!isSupabaseConfigured()) return;
    try {
      const { data, error } = await getInventori();
      if (error) throw error;
      setInventoriList(data || []);
    } catch (error) {
      console.error('Error fetching inventori:', error);
      toast({
        title: "Error",
        description: "Gagal mengambil data inventori",
        variant: "destructive",
      });
    }
  };

  const handleBarangSelect = (barangId) => {
    const barang = inventoriList.find(item => item.id === barangId);
    setSelectedBarang(barang);
    setFormData(prev => ({
      ...prev,
      barang_id: barangId,
      harga_satuan: barang?.harga || '',
      total_harga: prev.jumlah ? (parseFloat(prev.jumlah) * parseFloat(barang?.harga || 0)).toFixed(2) : ''
    }));
  };

  const handleJumlahChange = (e) => {
    const jumlah = e.target.value;
    setFormData(prev => ({
      ...prev,
      jumlah,
      total_harga: jumlah && prev.harga_satuan
        ? (parseFloat(jumlah) * parseFloat(prev.harga_satuan)).toFixed(2)
        : ''
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();

    if (!isSupabaseConfigured()) {
      toast({
        title: "Error",
        description: "Supabase belum dikonfigurasi",
        variant: "destructive",
      });
      return;
    }

    if (!formData.barang_id || !formData.jumlah || !formData.tanggal_transaksi) {
      toast({
        title: "Error",
        description: "Semua field harus diisi",
        variant: "destructive",
      });
      return;
    }

    if (parseInt(formData.jumlah) <= 0) {
      toast({
        title: "Error",
        description: "Jumlah harus lebih dari 0",
        variant: "destructive",
      });
      return;
    }

    if (selectedBarang && parseInt(formData.jumlah) > selectedBarang.stok) {
      toast({
        title: "Error",
        description: `Stok tidak mencukupi. Stok tersedia: ${selectedBarang.stok}`,
        variant: "destructive",
      });
      return;
    }

    setLoading(true);

    try {
      const { data, error } = await createTransaksi({
        barang_id: formData.barang_id,
        jumlah: parseInt(formData.jumlah),
        harga_satuan: parseFloat(formData.harga_satuan),
        total_harga: parseFloat(formData.total_harga),
        tanggal_transaksi: new Date(formData.tanggal_transaksi).toISOString()
      });

      if (error) throw error;

      toast({
        title: "Berhasil!",
        description: "Transaksi berhasil dicatat dan stok diperbarui",
      });

      // Reset form
      setFormData({
        barang_id: '',
        jumlah: '',
        harga_satuan: '',
        total_harga: '',
        tanggal_transaksi: new Date().toISOString().slice(0, 16)
      });
      setSelectedBarang(null);

      // Refresh inventori list
      await fetchInventori();

      if (onSuccess) {
        onSuccess();
      }
    } catch (error) {
      console.error('Error creating transaksi:', error);
      toast({
        title: "Error",
        description: error.message || "Gagal menyimpan transaksi",
        variant: "destructive",
      });
    } finally {
      setLoading(false);
    }
  };

  const formatRupiah = (amount) => {
    return new Intl.NumberFormat('id-ID', {
      style: 'currency',
      currency: 'IDR',
      minimumFractionDigits: 0,
    }).format(amount);
  };

  return (
    <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
      <CardHeader>
        <CardTitle className="text-gray-900">Catat Transaksi Baru</CardTitle>
        <CardDescription>
          Pilih barang dan jumlah penjualan
        </CardDescription>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="barang_id" className="text-gray-900">Pilih Barang</Label>
            <Select
              value={formData.barang_id}
              onValueChange={handleBarangSelect}
              disabled={loading}
            >
              <SelectTrigger id="barang_id" className="bg-white text-gray-900">
                <SelectValue placeholder="Pilih barang..." />
              </SelectTrigger>
              <SelectContent>
                {inventoriList.map((item) => (
                  <SelectItem key={item.id} value={item.id}>
                    {item.nama_barang} - Stok: {item.stok} - {formatRupiah(item.harga)}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            {selectedBarang && selectedBarang.stok < 10 && (
              <p className="text-sm text-yellow-600">
                ⚠️ Stok tinggal {selectedBarang.stok}
              </p>
            )}
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="jumlah" className="text-gray-900">Jumlah</Label>
              <Input
                id="jumlah"
                name="jumlah"
                type="number"
                placeholder="0"
                value={formData.jumlah}
                onChange={handleJumlahChange}
                disabled={loading || !formData.barang_id}
                className="bg-white text-gray-900 placeholder:text-gray-400"
                min="1"
                required
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="harga_satuan" className="text-gray-900">Harga Satuan (Rp)</Label>
              <Input
                id="harga_satuan"
                name="harga_satuan"
                type="number"
                value={formData.harga_satuan}
                disabled
                className="bg-gray-100 text-gray-900"
                readOnly
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="total_harga" className="text-gray-900">Total Harga</Label>
            <div className="relative">
              <Input
                id="total_harga"
                name="total_harga"
                type="text"
                value={formData.total_harga ? formatRupiah(formData.total_harga) : ''}
                disabled
                className="bg-gray-100 text-gray-900 font-semibold text-lg"
                readOnly
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="tanggal_transaksi" className="text-gray-900">Tanggal & Waktu</Label>
            <Input
              id="tanggal_transaksi"
              name="tanggal_transaksi"
              type="datetime-local"
              value={formData.tanggal_transaksi}
              onChange={(e) => setFormData(prev => ({ ...prev, tanggal_transaksi: e.target.value }))}
              disabled={loading}
              className="bg-white text-gray-900"
              required
            />
          </div>

          <Button
            type="submit"
            disabled={loading}
            className="w-full bg-blue-600 hover:bg-blue-700 text-white"
          >
            {loading ? (
              <>
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                Menyimpan...
              </>
            ) : (
              <>
                <ShoppingCart className="mr-2 h-4 w-4" />
                Catat Transaksi
              </>
            )}
          </Button>
        </form>
      </CardContent>
    </Card>
  );
};

export default TransaksiForm;
