
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useToast } from '@/components/ui/use-toast';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import { getInventori, deleteInventori, isSupabaseConfigured } from '@/lib/supabase';
import { Loader2, Pencil, Trash2, Package } from 'lucide-react';
import { motion } from 'framer-motion';

const InventoriTable = ({ onEdit, refreshTrigger }) => {
  const { toast } = useToast();
  const [loading, setLoading] = useState(true);
  const [inventori, setInventori] = useState([]);
  const [deleteDialog, setDeleteDialog] = useState({ open: false, barang: null });
  const [deleting, setDeleting] = useState(false);

  const fetchInventori = async () => {
    setLoading(true);
    try {
      if (!isSupabaseConfigured()) {
        throw new Error("Supabase belum dikonfigurasi");
      }

      const { data, error } = await getInventori();
      
      if (error) throw error;
      
      setInventori(data || []);
    } catch (error) {
      console.error('Error fetching inventori:', error);
      // Only show toast if it's not the initial configuration error to avoid spamming
      if (error.message !== "Supabase belum dikonfigurasi") {
        toast({
          title: "Error",
          description: "Gagal mengambil data inventori",
          variant: "destructive",
        });
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchInventori();
  }, [refreshTrigger]);

  const handleDelete = async () => {
    if (!deleteDialog.barang) return;

    setDeleting(true);
    try {
      const { error } = await deleteInventori(deleteDialog.barang.id);
      
      if (error) throw error;

      toast({
        title: "Berhasil!",
        description: "Barang berhasil dihapus dari inventori",
      });

      setDeleteDialog({ open: false, barang: null });
      fetchInventori();
    } catch (error) {
      console.error('Error deleting inventori:', error);
      toast({
        title: "Error",
        description: error.message || "Gagal menghapus barang",
        variant: "destructive",
      });
    } finally {
      setDeleting(false);
    }
  };

  const formatRupiah = (amount) => {
    return new Intl.NumberFormat('id-ID', {
      style: 'currency',
      currency: 'IDR',
      minimumFractionDigits: 0,
    }).format(amount);
  };

  const formatDate = (dateString) => {
    return new Date(dateString).toLocaleDateString('id-ID', {
      day: 'numeric',
      month: 'short',
      year: 'numeric',
    });
  };

  if (loading) {
    return (
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardContent className="flex items-center justify-center py-12">
          <Loader2 className="h-8 w-8 animate-spin text-blue-600" />
        </CardContent>
      </Card>
    );
  }

  if (inventori.length === 0) {
    return (
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardContent className="flex flex-col items-center justify-center py-12">
          <Package className="h-12 w-12 text-gray-400 mb-4" />
          <p className="text-gray-600 text-center">Belum ada barang di inventori</p>
          <p className="text-gray-400 text-sm text-center mt-2">
            Tambahkan barang pertama Anda untuk memulai
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <>
      <Card className="bg-white/95 backdrop-blur-sm border-blue-200">
        <CardHeader>
          <CardTitle className="text-gray-900">Daftar Inventori</CardTitle>
          <CardDescription>
            Total {inventori.length} barang terdaftar
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 text-sm font-semibold text-gray-900">Nama Barang</th>
                  <th className="text-right py-3 px-4 text-sm font-semibold text-gray-900">Harga</th>
                  <th className="text-center py-3 px-4 text-sm font-semibold text-gray-900">Stok</th>
                  <th className="text-left py-3 px-4 text-sm font-semibold text-gray-900">Ditambahkan</th>
                  <th className="text-center py-3 px-4 text-sm font-semibold text-gray-900">Aksi</th>
                </tr>
              </thead>
              <tbody>
                {inventori.map((item, index) => (
                  <motion.tr
                    key={item.id}
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: index * 0.05 }}
                    className="border-b border-gray-100 hover:bg-blue-50/50 transition-colors"
                  >
                    <td className="py-3 px-4 text-gray-900">{item.nama_barang}</td>
                    <td className="py-3 px-4 text-right text-gray-900 font-medium">
                      {formatRupiah(item.harga)}
                    </td>
                    <td className="py-3 px-4 text-center">
                      <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                        item.stok === 0 
                          ? 'bg-red-100 text-red-700'
                          : item.stok < 10
                          ? 'bg-yellow-100 text-yellow-700'
                          : 'bg-green-100 text-green-700'
                      }`}>
                        {item.stok}
                      </span>
                    </td>
                    <td className="py-3 px-4 text-gray-600 text-sm">
                      {formatDate(item.created_at)}
                    </td>
                    <td className="py-3 px-4">
                      <div className="flex items-center justify-center gap-2">
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={() => onEdit(item)}
                          className="h-8 text-blue-600 hover:text-blue-700 hover:bg-blue-50"
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={() => setDeleteDialog({ open: true, barang: item })}
                          className="h-8 text-red-600 hover:text-red-700 hover:bg-red-50"
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </td>
                  </motion.tr>
                ))}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>

      <Dialog open={deleteDialog.open} onOpenChange={(open) => setDeleteDialog({ open, barang: null })}>
        <DialogContent className="bg-white">
          <DialogHeader>
            <DialogTitle className="text-gray-900">Hapus Barang?</DialogTitle>
            <DialogDescription>
              Apakah Anda yakin ingin menghapus <span className="font-semibold text-gray-900">{deleteDialog.barang?.nama_barang}</span> dari inventori?
              Tindakan ini tidak dapat dibatalkan.
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setDeleteDialog({ open: false, barang: null })}
              disabled={deleting}
              className="border-gray-300 text-gray-900"
            >
              Batal
            </Button>
            <Button
              onClick={handleDelete}
              disabled={deleting}
              className="bg-red-600 hover:bg-red-700 text-white"
            >
              {deleting ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Menghapus...
                </>
              ) : (
                'Hapus'
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
};

export default InventoriTable;
