import React from 'react';
import { motion } from 'framer-motion';
import { TrendingUp, DollarSign, ShoppingCart, Users } from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';

const Dashboard = ({ transactions }) => {
  const totalRevenue = transactions.reduce((sum, t) => sum + t.total, 0);
  const totalTransactions = transactions.length;
  const totalItems = transactions.reduce((sum, t) => 
    sum + t.items.reduce((itemSum, item) => itemSum + item.quantity, 0), 0
  );
  const uniqueCustomers = new Set(transactions.map(t => t.customerName)).size;

  const stats = [
    {
      title: 'Total Pendapatan',
      value: `Rp ${totalRevenue.toLocaleString('id-ID')}`,
      icon: DollarSign,
      color: 'text-green-400',
      bgColor: 'bg-green-900/30',
    },
    {
      title: 'Total Transaksi',
      value: totalTransactions,
      icon: ShoppingCart,
      color: 'text-blue-400',
      bgColor: 'bg-blue-900/30',
    },
    {
      title: 'Total Item Terjual',
      value: totalItems,
      icon: TrendingUp,
      color: 'text-purple-400',
      bgColor: 'bg-purple-900/30',
    },
    {
      title: 'Pelanggan Unik',
      value: uniqueCustomers,
      icon: Users,
      color: 'text-orange-400',
      bgColor: 'bg-orange-900/30',
    },
  ];

  const paymentMethodStats = transactions.reduce((acc, t) => {
    acc[t.paymentMethod] = (acc[t.paymentMethod] || 0) + 1;
    return acc;
  }, {});

  return (
    <motion.div
      initial={{ opacity: 0, scale: 0.95 }}
      animate={{ opacity: 1, scale: 1 }}
      transition={{ duration: 0.3 }}
      className="space-y-6"
    >
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        {stats.map((stat, index) => (
          <motion.div
            key={stat.title}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.3, delay: index * 0.1 }}
          >
            <Card className="bg-slate-800/50 backdrop-blur-sm border-slate-700 hover:border-blue-500/50 transition-all">
              <CardContent className="p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-slate-400 text-sm mb-1">{stat.title}</p>
                    <p className={`text-2xl font-bold ${stat.color}`}>
                      {stat.value}
                    </p>
                  </div>
                  <div className={`p-3 rounded-lg ${stat.bgColor}`}>
                    <stat.icon className={`w-6 h-6 ${stat.color}`} />
                  </div>
                </div>
              </CardContent>
            </Card>
          </motion.div>
        ))}
      </div>

      <Card className="bg-slate-800/50 backdrop-blur-sm border-slate-700">
        <CardHeader>
          <CardTitle className="text-xl text-white">Metode Pembayaran</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {Object.entries(paymentMethodStats).map(([method, count]) => (
              <div key={method} className="flex items-center justify-between">
                <span className="text-slate-300 capitalize">
                  {method === 'cash' ? 'Tunai' : method === 'debit' ? 'Debit' : 'QRIS'}
                </span>
                <div className="flex items-center gap-3">
                  <div className="w-32 bg-slate-700 rounded-full h-2">
                    <div
                      className="bg-blue-500 h-2 rounded-full transition-all"
                      style={{ width: `${(count / totalTransactions) * 100}%` }}
                    />
                  </div>
                  <span className="text-blue-300 font-semibold w-12 text-right">
                    {count}
                  </span>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </motion.div>
  );
};

export default Dashboard;