
import React, { useState, useEffect } from 'react';
import { AnimatePresence } from 'framer-motion';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '@/contexts/AuthContext';
import { Button } from '@/components/ui/button';
import { UserCircle } from 'lucide-react';
import MenuPage from '@/pages/MenuPage';
import InventoriPage from '@/pages/InventoriPage';
import TransaksiPage from '@/pages/TransaksiPage';
import PelangganPage from '@/pages/PelangganPage';
import { useToast } from '@/components/ui/use-toast';

const CashierApp = () => {
  const [currentPage, setCurrentPage] = useState('menu');
  const { user } = useAuth();
  const navigate = useNavigate();
  const { toast } = useToast();

  const checkAccess = (module) => {
    // Admin has access to everything
    if (user?.role === 'admin') return true;
    // Check specific module access
    const permission = user?.access?.find(a => a.modul === module);
    return permission?.akses === true;
  };

  const navigateTo = (page) => {
    if (page === 'menu') {
      setCurrentPage('menu');
      return;
    }

    if (!checkAccess(page)) {
      toast({
        title: "Akses Ditolak",
        description: `Anda tidak memiliki izin untuk mengakses modul ${page}`,
        variant: "destructive"
      });
      return;
    }

    setCurrentPage(page);
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const goBack = () => {
    setCurrentPage('menu');
  };

  const renderPage = () => {
    switch (currentPage) {
      case 'inventori':
        return <InventoriPage onBack={goBack} />;
      case 'transaksi':
        return <TransaksiPage onBack={goBack} />;
      case 'pelanggan':
        return <PelangganPage onBack={goBack} />;
      case 'menu':
      default:
        return <MenuPage onNavigate={navigateTo} />;
    }
  };

  return (
    <div className="container mx-auto px-4 py-8 max-w-5xl">
      {/* Header Info */}
      <div className="flex justify-between items-center mb-6 text-white/80">
        <div className="text-sm">
          Logged in as: <span className="font-semibold text-white">{user?.username}</span> ({user?.role})
        </div>
        <Button variant="ghost" size="sm" onClick={() => navigate('/profile')} className="hover:bg-white/10 text-white">
          <UserCircle className="mr-2 h-4 w-4" /> Profil
        </Button>
      </div>

      <AnimatePresence mode="wait">
        {renderPage()}
      </AnimatePresence>
    </div>
  );
};

export default CashierApp;
